/*
 *  b20func.c - Support functions for parsing PIC graphic
 *              format state diagram files to yield VHDL.
 *
 *  Tom Mayo - 6-15-95
 */

#include "b20.h"

/*
 *  preamble - version, license, etc...
 */
void preamble(FILE *fp)
{
    fprintf(fp, version);
}


/*
 *  usage - provide a crib sheet of switches.
 */

void usage()
{
    fprintf(stderr, "usage: brusey20 [options]\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "       brusey20 parses a PIC file from infile and\n");
    fprintf(stderr, "       generates VHDL code on outfile with\n");
    fprintf(stderr, "       debug information on standard error.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "       -h         = help.\n");
    fprintf(stderr, "       -l         = license statement.\n");
    fprintf(stderr, "       -i=infile  = use infile as PIC input.\n");
    fprintf(stderr, "       -o=outfile = use outfile as VHDL output.\n");
    fprintf(stderr, "       -da        = turn all debugging on.\n");
    fprintf(stderr, "       -dp        = turn PIC parse debugging on.\n");
    fprintf(stderr, "       -df        = turn data structure fill debugging on.\n");
    fprintf(stderr, "       -de        = turn expression parse debugging on.\n");
    fprintf(stderr, "       -di        = turn I/O find debugging on.\n");
    fprintf(stderr, "       -dv        = turn VHDL code generation debugging on.\n");
    fprintf(stderr, "      [-sr        = reset synchronously.]\n");
    fprintf(stderr, "      [-so        = synchronize all Moore outputs.]\n");
    fprintf(stderr, "      [-ve        = generate explicit default state "
        "transitions.]\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "        [] => not yet implemented.\n");
    
    exit(1);
}

/*
 *  fatal_error - function to print a fatal error
 *    message then bug out.
 */

void fatal_error(char *s)
{
  int i;
  FILE *fp;

  for (i = 0; i < 2; i++)
  {
    if (i == 0)
      fp = outfp;
    else
      fp = stderr;

    fprintf(fp, "-- FATAL ERROR:\n");
    fprintf(fp, "%s\n", s);
    fprintf(fp, "--   Please contact the author with information\n");
    fprintf(fp, "--   about the events leading to this misfortune.\n");
  }

  exit(1);
}

/*
 *  create_string - function to create a string structure
 *    to store label information from .PIC input.
 */

string_s_t *create_string(void)
{
  string_s_t *created;
  created = (string_s_t *)malloc(sizeof(string_s_t));

  if (created == (string_s_t *)NULL)
  {
		fatal_error("--   cannot create string structure.");
	}

	created->next_overall = (string_s_t *)NULL;
	created->next_this_state = (string_s_t *)NULL;

	return(created);
};

/*
 *  create_transition - function to create a transition structure
 *    to store transition information.
 */

transition_s_t *create_transition(void)
{
	transition_s_t *created;
	created = (transition_s_t *)malloc(sizeof(transition_s_t));

	if (created == (transition_s_t *)NULL)
	{
		fatal_error("--   cannot create transition structure.");
	}

	created->from_state = (state_s_t *)NULL;
	created->to_state = (state_s_t *)NULL;
	created->condition = (string_s_t *)NULL;
  created->priority = 99;
	created->next_overall = (transition_s_t *)NULL;
	created->next_this_state = (transition_s_t *)NULL;

	return(created);
};

/*
 *  create_state - function to create a state structure
 *    to store state information.
 */

state_s_t *create_state(void)
{
	state_s_t *created;
	created = (state_s_t *)malloc(sizeof(state_s_t));

	if (created == (state_s_t *)NULL)
	{
		fatal_error("--   cannot create state structure.");
	}

	created->name = (string_s_t *)NULL;
	created->first_transition = (transition_s_t *)NULL;
	created->first_output = (string_s_t *)NULL;
	created->next_overall = (state_s_t *)NULL;

	return(created);
};

/*
 *  fill_transition - function to add a new transition
 *    to the linked list and store geometric information
 *    about the transtion.
 */

void fill_transition(xy_s_t *center_xy, xy_s_t *from_xy, xy_s_t *to_xy)
{
    extern transition_s_t *current_transition, *first_transition;
    transition_s_t *new_transition;

    if (current_transition == (transition_s_t *)NULL)
    {
        first_transition = create_transition();
        current_transition = first_transition;
    }
    else
    {
        new_transition = create_transition();
        current_transition->next_overall = new_transition;
        current_transition = new_transition;
    }

    current_transition->center_xy.x = center_xy->x;
    current_transition->center_xy.y = center_xy->y;
    current_transition->from_xy.x = from_xy->x;
    current_transition->from_xy.y = from_xy->y;
    current_transition->to_xy.x = to_xy->x;
    current_transition->to_xy.y = to_xy->y;
    current_transition->seq_num = transitions;
    current_transition->next_overall = (transition_s_t *)NULL;

    if (debug_parse)
    {
        fprintf(stderr, "-- Transition at (%2.3f, %2.3f): "
			"(%2.3f, %2.3f) -> (%2.3f, %2.3f).\n",
			current_transition->center_xy.x,
			current_transition->center_xy.y,
			current_transition->from_xy.x,
			current_transition->from_xy.y,
			current_transition->to_xy.x,
			current_transition->to_xy.y);
    }

	transitions++;
}

/*
 *  line_center - function to find the center point
 *    for given line start and end points.
 */

xy_s_t *line_center(xy_s_t *start_xy, xy_s_t *end_xy)
{
	static xy_s_t center_xy;

	center_xy.x = (start_xy->x + end_xy->x) / 2.0;
	center_xy.y = (start_xy->y + end_xy->y) / 2.0;

	return(&center_xy);
}

/*
 *  arc_center - pain in the ass function to find
 *    the center point for given arc start, end,
 *    and origin points, as well as rotation
 *    direction.
 */

xy_s_t *arc_center(xy_s_t *start_xy, xy_s_t *end_xy,
	xy_s_t *origin_xy, int rotation)
{
	double radius, theta_start, theta_end, theta_center;
	static xy_s_t center_xy;

	radius = (distance(start_xy, origin_xy) +
		distance(end_xy, origin_xy)) / 2.0;
	
	theta_start = atan2(start_xy->y - origin_xy->y,
		start_xy->x - origin_xy->x);

	theta_end = atan2(end_xy->y - origin_xy->y,
		end_xy->x - origin_xy->x);

	if (rotation == ARC_CCW && theta_end < theta_start)
		theta_end += 2 * PI;

	if (rotation == ARC_CW && theta_end > theta_start)
		theta_end -= 2 * PI;

	theta_center = (theta_start + theta_end) / 2.0;

	center_xy.x = radius * cos(theta_center) + origin_xy->x;
	center_xy.y = radius * sin(theta_center) + origin_xy->y;

	return(&center_xy);
}

/*
 *  distance - function to compute the linear distance
 *    between two points.
 */

double distance(xy_s_t *start_xy, xy_s_t *end_xy)
{
	double r;

	r = sqrt(pow(start_xy->x - end_xy->x, 2.0) +
		pow(start_xy->y - end_xy->y, 2.0));

/*
  fprintf(stderr, "-- Points (%f,%f) and (%f,%f) distance %f.\n",
		start_xy->x, start_xy->y,
		end_xy->x, end_xy->y,
		r);
*/

	return(r);
}

/*
 *  fill_state - function to add a new state to the
 *    linked list and store geometric information
 *    about the state.
 */

void fill_state(xy_s_t *center_xy, double radius)
{
    extern state_s_t *current_state, *first_state;
    state_s_t *new_state;

    if (current_state == (state_s_t *)NULL)
    {
        first_state = create_state();
        current_state = first_state;
    }
    else
    {
        new_state = create_state();
        current_state->next_overall = new_state;
        current_state = new_state;
    }

    current_state->center_xy.x = center_xy->x;
    current_state->center_xy.y = center_xy->y;
    current_state->radius = radius;
    current_state->seq_num = states;
    current_state->next_overall = (state_s_t *)NULL;

    if (debug_parse)
    {
        fprintf(stderr, "-- State at (%2.3f, %2.3f): radius %2.3f.\n",
			current_state->center_xy.x,
			current_state->center_xy.y,
			current_state->radius);
    }

	states++;
}

/*
 *  fill_string - function to add a new string to the
 *    linked list and store geometric and text information
 *    about the string.
 */

void fill_string(xy_s_t *origin_xy, just_t just, char *text)
{
    extern string_s_t *current_string, *first_string;
    string_s_t *new_string;

    if (current_string == (string_s_t *)NULL)
    {
        first_string = create_string();
        current_string = first_string;
    }
    else
    {
        new_string = create_string();
        current_string->next_overall = new_string;
        current_string = new_string;
    }

    current_string->origin_xy.x = origin_xy->x;
    current_string->origin_xy.y = origin_xy->y;
    current_string->just = just;
	current_string->text = (char *)malloc(strlen(text) + 1);
	strcpy(current_string->text, text);
	current_string->next_overall = (string_s_t *)NULL;

    if (debug_parse)
    {
        fprintf(stderr, "-- String at (%2.3f, %2.3f): \"%s\".\n",
			current_string->origin_xy.x,
			current_string->origin_xy.y,
			current_string->text);
    }

	strings++;
}

/*
 *  add_input and add_output - functions to update lists of
 *    inputs and outputs to be used in the VHDL ENTITY description
 *    and PROCESS sensitivity list.
 */

/*
 *  The second parm to the following functions is a flag
 *  which allows the program to: 
 *
 *  place the proper inputs on the sensitivity list of the
 *  registered process to allow asynchronous resetting of
 *  the machine.
 *
 *  or create registered outputs.
 */

void add_input(char *s, int async)
{
	io_s_t *tmp_in;
	int found;
	char errstring[LARGE_STRING];

/*
 *  Don't panic, this is just in case.
 */
	sprintf(errstring,
		"--   Out of memory for input signal name \"%s\".", s);
	if (current_in == (io_s_t *)NULL)
	{
		if ((first_in = current_in =
			(io_s_t *)malloc(sizeof(io_s_t))) == (io_s_t *)NULL)
		{
			fatal_error(errstring);
		}

		if ((first_in->name =
			(char *)malloc(strlen(s) + 1)) == (char *)NULL)
		{
			fatal_error(errstring);
		}

		strcpy(first_in->name, s);
		first_in->async = async;
		first_in->next = (io_s_t *)NULL;
	}
	else
/*
 *  This is not the first, add to the linked list.
 */
	{
		tmp_in = first_in;
		found = 0;

		while (tmp_in != (io_s_t *)NULL)
		{
			if ((found = !strcmp(tmp_in->name, s)))
			{
				if (async != tmp_in->async)
				{
					sprintf(errstring,
						"--   Input %s has been fed to an asynchronous\n"
						"--   register input and should not be used\n"
						"--   for transitions.", s);
					fatal_error(errstring);
				}

				break;
			}

			tmp_in = tmp_in->next;
		}
			
		if (!found)
		{
			if (debug_iofind)
    			fprintf(stderr, "-- Adding input signal %s.\n", s);

			if ((tmp_in = (io_s_t *)malloc(sizeof(io_s_t))) ==
				(io_s_t *)NULL)
			{
				fatal_error(errstring);
			}

			if ((tmp_in->name = (char *)malloc(strlen(s) + 1)) ==
				(char *)NULL)
					fatal_error(errstring);

			strcpy(tmp_in->name, s);
			tmp_in->async = async;
			tmp_in->next = (io_s_t *)NULL;

			current_in->next = tmp_in;
			current_in = tmp_in;
		}
	}
}

void add_output(char *s, int async)
{
	io_s_t *tmp_out;
	int found;
	char errstring[LARGE_STRING];

/*
 *  Don't panic, this is just in case.
 */
	sprintf(errstring,
		"--   Out of memory for output signal name \"%s\".", s);

/*
 *  Check to see if this is the first output ever.
 */
	if (current_out == (io_s_t *)NULL)
	{
		if ((first_out = current_out =
			(io_s_t *)malloc(sizeof(io_s_t))) == (io_s_t *)NULL)
		{
			fatal_error(errstring);
		}

		if ((first_out->name =
			(char *)malloc(strlen(s) + 1)) == (char *)NULL)
		{
			fatal_error(errstring);
		}

		strcpy(first_out->name, s);
		first_out->async = async;
		first_out->next = (io_s_t *)NULL;
	}

/*
 *  This is not the first, add to the linked list.
 */
	else
	{
		tmp_out = first_out;
		found = 0;

		while (tmp_out != (io_s_t *)NULL)
		{
			if ((found = !strcmp(tmp_out->name, s)))
			{
/*
 *  If the output ever appears as synchronous, it must
 *  always be synchronous.
 */
				tmp_out->async &= async;

				break;
			}

			tmp_out = tmp_out->next;
		}
			
		if (!found)
		{
			if (debug_iofind)
	    		fprintf(stderr, "-- Adding output signal %s.\n", s);

			if ((tmp_out = (io_s_t *)malloc(sizeof(io_s_t))) ==
				(io_s_t *)NULL)
			{
				fatal_error(errstring);
			}

			if ((tmp_out->name = (char *)malloc(strlen(s) + 1)) ==
				(char *)NULL)
					fatal_error(errstring);

			strcpy(tmp_out->name, s);
			tmp_out->async = async;
			tmp_out->next = (io_s_t *)NULL;

			current_out->next = tmp_out;
			current_out = tmp_out;
		}
	}
}

void set_priority(int p)
{
	prio = p;
}

/*
 *  sift_assignment - function to sift through a transition
 *    or output string and print out all the output assignments
 *    in a nice form.
 */

void sift_assignment(string_s_t *string, int sift_type)
{
	char my_text[LARGE_STRING], *ptr1, *ptr2;

/*
 *  Assignments must end in a semi-colon, expressions must
 *    not.
 */
 	if (string == (string_s_t *)NULL ||
		string->text == (char *)NULL ||
 		*(string->text + strlen(string->text) - 1) != ';')
	{
		if (debug_vhdl)
			fprintf(stderr, "-- string contains no assignments.\n");
		return;
	}

/*
 *  Ditch the leading expression if any.
 */
	if ((ptr1 = strchr(string->text, '|')) != (char *)NULL)
		strcpy(my_text, (ptr1 + 1));
	else
		strcpy(my_text, string->text);
		
	ptr2 = strtok(my_text, ";");

/*
 *  Sift through each output assignment.
 */
	while (ptr2 != (char *)NULL)
	{
		while (*ptr2 == ' ')
			ptr2++;

		if (sift_type & SIFT_INDENT_FAR)
			fprintf(outfp, "  ");

		if (sift_type & SIFT_ADD_NEXT)
				fprintf(outfp, "        next_%s;\n", ptr2);
		else
				fprintf(outfp, "        %s;\n", ptr2);

		ptr2 = strtok((char *)NULL, ";");
	}
}

/*
 *  sift_expression - function to sift through a transition
 *    and output the transition expression and any assignments.
 *
 *  Notes:  returns SIFT_DEFAULT if processing of this
 *          transition should be deferred because it is 
 *          the default.
 *
 *          call with sift_type = SIFT_DEFAULT to force
 *          processing of the default transition.  Call
 *          with SIFT_CAP to finish of the state actions.
 *          In both cases, transition is ignored.
 */

int sift_expression(transition_s_t *transition, int sift_type)
{
	char my_text[LARGE_STRING], *ptr;
	static int ifs = 0;

	switch(sift_type)
	{
		case SIFT_RESET:
		{
/*
 *  When a new state is found, reset the "ifs" counter.
 */
			ifs = 0;
			return(SIFT_NORMAL);

			break;
		}

/*
 *  For the default transition, processing is different than
 *  for other transitions.
 */
 		case SIFT_DEFAULT:
		{
			if (transition == (transition_s_t *)NULL)
			{
				return(SIFT_NORMAL);
			}

			if (ifs)
			{
				fprintf(outfp, "        ELSE\n");
				sift_assignment(transition->condition,
					SIFT_INDENT_FAR | SIFT_ADD_NEXT);
				fprintf(outfp, "          next_state <= %s;\n",
					transition->to_state->name->text);
			}
			else
			{
				sift_assignment(transition->condition,
					SIFT_ADD_NEXT);
				fprintf(outfp, "        next_state <= %s;\n",
					transition->to_state->name->text);
			}

			return(SIFT_NORMAL);

			break;
		}

/*
 *  Put and END IF in if there have been
 *    transition expressions.
 */
 		case SIFT_CAP:
		{
			if (ifs)
				fprintf(outfp, "        END IF;\n");

			return(SIFT_NORMAL);

			break;
		}

		default:
		{

			if (transition == (transition_s_t *)NULL)
			{
				fatal_error(
					"--   sift_expression(0, SIFT_NORMAL) was\n"
					"--   unexpectedly called with a null transition.\n");
			}

/*
 *  If this is the default transition, do nothing but
 *    report it upon return.
 */
		    if (transition->condition == (string_s_t *)NULL ||
				transition->condition->text == (char *)NULL ||
    			*(transition->condition->text) == '|' ||
        		!strlen(transition->condition->text))
			{
				if (debug_vhdl)
					fprintf(stderr, "-- default transition, deferring.\n");

				return(SIFT_DEFAULT);
			}

			if (transition->condition->text == (char *)NULL)
			{
				fatal_error(
					"--   sift_expression(0, SIFT_NORMAL) was\n"
					"--   unexpectedly called with null condition text.");
			}

/*
 *  Process a transition with a condition specified.
 */
			if ((ptr = strchr((transition->condition)->text, ':')) != (char *)NULL)
				strcpy(my_text, ++ptr);
			else
				strcpy(my_text, (transition->condition)->text);

			if (ifs)
				fprintf(outfp, "        ELS");
			else
				fprintf(outfp, "        ");

/*
 *  Just want the expression part.
 */
			if ((ptr = strtok(my_text, "|")) != (char *)NULL)
			{
				while (*ptr == ' ')
					ptr++;
				fprintf(outfp, "IF ( %s ) THEN\n", ptr);
				sift_assignment(transition->condition,
					SIFT_INDENT_FAR | SIFT_ADD_NEXT);
				fprintf(outfp, "          next_state <= %s;\n",
					transition->to_state->name->text);
				ifs++;
			}

			break;
		}
	}

	return(SIFT_NORMAL);
}

/*
 *  It would be nice to use the following sort of scheme,
 *  but exit() will take care of this in a pinch.
 */

/*
 *  destroy_string - function to free the memory associated
 *    with a string structure.
 */

void destroy_string(string_s_t *destroy)
{
	free(destroy->text);
	free(destroy);
}

/*
 *  destroy_transition - function to free the memory associated
 *    with a transition structure.
 */

void destroy_transition(transition_s_t *destroy)
{
	free(destroy);
}

/*
 *  destroy_state - function to free the memory associated
 *    with a state structure.
 */

void destroy_state(state_s_t *destroy)
{
	free(destroy);
}
